# %% [markdown]
# # KRIGING METAMODELING: TRUSS DATA SET
# 
# This example showcases how to perform Kriging metamodeling using existing data sets.
# The data sets come from a finite element model of a truss structure and are retrieved from different MAT-files.
# The files consist of an experimental design of size $200$ and a validation set of size $10^4$.

# %% [markdown]
# ## Package imports

# %%
from uqpylab import sessions, display_util
import numpy as np
import matplotlib.pyplot as plt
import scipy.io

# %% [markdown]
# ## Initialize common plotting parameters

# %%
display_util.load_plt_defaults()
uq_colors = display_util.get_uq_color_order()

# %% [markdown]
# ## Start a remote UQCloud session

# %%
# Start the session
mySession = sessions.cloud()
# (Optional) Get a convenient handle to the command line interface
uq = mySession.cli
# Reset the session
mySession.reset()

# %% [markdown]
# ## Set random seed for reproducibility

# %%
uq.rng(0,'twister');

# %% [markdown]
# ## Retrieve data sets

# %% [markdown]
# The experimental design and the validation basis are stored in two separate files

# %%
# Read the binary files (stored in .mat format)
mat = scipy.io.loadmat('Truss_Experimental_Design.mat')
X = mat['X']
Y = mat['Y']

# Also read the validation basis data set file and store the contents in matrices:
mat = scipy.io.loadmat('Truss_Validation_Basis.mat')
X_val = mat['Xval']
Y_val = mat['Yval']

# %% [markdown]
# ## Kriging metamodel

# %% [markdown]
# Select Kriging as the metamodeling tool:

# %%
MetaOpts = {
    'Type': 'Metamodel',
    'MetaType': 'Kriging'
}

# %% [markdown]
# Use experimental design loaded from the data files:

# %%
MetaOpts['ExpDesign'] = {
    'X': X.tolist(),
    'Y': Y.tolist()
}

# %% [markdown]
# Use maximum-likelihood to estimate the hyperparameters:

# %%
MetaOpts['EstimMethod'] = 'ML'

# %% [markdown]
# Use the built-in covariance-matrix adaptation evolution strategy optimization algorithm (CMAES) for estimating the hyperparameters:

# %%
MetaOpts['Optim'] = {
    'Method': 'CMAES'
}

# %% [markdown]
# Provide the validation data set to get the validation error:

# %%
MetaOpts['ValidationSet'] = {
    'X': X_val.tolist(),
    'Y': Y_val.tolist()
}

# %% [markdown]
# Create the Kriging metamodel:

# %%
myKriging = uq.createModel(MetaOpts)

# %% [markdown]
# Print a summary of the resulting Kriging metamodel:

# %%
uq.print(myKriging)

# %% [markdown]
# ## Validation

# %% [markdown]
# Evaluate the Kriging metamodel at the validation set:

# %%
Y_Krg = uq.evalModel(myKriging, X_val)

# %% [markdown]
# Plot histograms of the true output and the Kriging prediction:

# %%
plt.hist(Y_val, color=uq_colors[0], alpha = 0.8)
plt.hist(Y_Krg, color=uq_colors[1], alpha = 0.8)
legend_text = ['True model response', 'Kriging prediction']
plt.legend(legend_text, frameon=False, loc="best")
plt.xlabel('$\\mathrm{Y}$')
plt.ylabel('Counts')
plt.show()

# %% [markdown]
# Plot the true vs. predicted values:

# %%
fig = plt.figure(figsize=(6, 6))
plt.scatter(Y_val, Y_Krg, marker='o', s=3)
plt.plot([np.min(Y_val), np.max(Y_val)], [np.min(Y_val), np.max(Y_val)], 'k')
plt.axis([np.min(Y_val), np.max(Y_val), np.min(Y_val), np.max(Y_val)])
plt.axis('equal')
plt.grid(True)
plt.xlabel('$\\mathrm{Y_{true}}$')
plt.ylabel('$\\mathrm{Y_{Krg}}$')
plt.show()

# %% [markdown]
# Print the validation and leave-one-out (LOO) cross-validation errors:

# %%
print('Kriging metamodel validation error: {:5.4e}'.format(myKriging['Error']['Val']))
print('Kriging metamodel LOO error:        {:5.4e}'.format(myKriging['Error']['LOO']))

# %% [markdown]
# ## Terminate the remote UQCloud session

# %%
mySession.quit()


